#include "JabberServiceProvider.h"
#include "jcfcoreutils/StringUtils.h"
#include "csfunified/services/interface/SystemService.h"
#include "csfunified/services/interface/ContactService.h"
#include "csfunified/services/interface/ConfigService.h"
#include "csfunified/services/interface/ConfigValue.h"
#include "csfunified/services/interface/Presence.h"
#include "csfunified/services/interface/PresenceService.h"
#include "csfunified/services/interface/PresenceServiceCapabilities.h"
#include "csfunified/services/interface/MeetingAuthenticatorIds.h"
#include "csfunified/services/interface/PresenceOption.h"
#include "csfunified/services/interface/Credentials.h"
#include "csfsystemmonitor/SystemMonitor.h"
#include "csfsystemmonitor/SystemMonitorFactory.h"
#include "boost/bind.hpp"
#include "boost/function.hpp"
#include "csfunified/framework/ServicesDispatcher.h"

#ifdef WIN32
#include "featuresets/adapters/IMMgrHelper.h"
#include "featuresets/adapters/Dependence.h"
#else
#include "sdkwrapper/mac/InstantMessageMgr.h"
#endif

#include "JabberServiceSink.h"

static CSFLogger* MeetingServiceLogger = CSFLogger_getLogger("MeetingService-MeetingServiceLogger");

//const static std::wstring WebEx_DS_Token = L"WebEx_DS_Token";
//const static std::wstring WebEx_DS_Url = L"WebEx_DS_Url";
//const static std::wstring CUP = L"CUP";
//const static std::wstring WBX = L"WebEx";
//
namespace CSFUnified
{
	JabberServiceProvider::JabberServiceProvider(void)
	{
	}

	JabberServiceProvider::~JabberServiceProvider(void)
	{

//		CSFSystemMonitor::SystemMonitorFactory::getInstance()->removeNetworkObserver(m_ptrJabberWerxCommonEvent);
		if (PropertyNotifierConnectionPtr l = m_currentPresenceChange.lock())
		{
			if (l->connected())
			{
				l->disconnect();
			}
		}
//		if(PropertyNotifierConnectionPtr l = m_ptrDSTokenChanged.lock())
//		{
//			if(l->connected())
//			{
//				l->disconnect();
//			}
//		}
	}
    
    JabberServiceProvider& JabberServiceProvider::getInstance()
    {
        static JabberServiceProvider s_provider;
        return s_provider;
    }
    
	void JabberServiceProvider::start()
	{
		initDefaultService();
	}

	void JabberServiceProvider::stop()
	{

	}

    void JabberServiceProvider::setUnifiedFactory(SMART_PTR_NS::shared_ptr<UnifiedFactory> unifiedFactory)
    {
        m_unifiedFactory = unifiedFactory;
		initDefaultService();
    }
    
    SMART_PTR_NS::shared_ptr<UnifiedFactory> JabberServiceProvider::getUnifiedFactory()
    {
        return m_unifiedFactory;
    }
    
    std::string JabberServiceProvider::getMyAccount()
    {
        std::string JID = "";
        
        do
        {
            if (NULL == m_contactService.get())
            {
                break;
            }
            
            SMART_PTR_NS::shared_ptr<Contact> pClientUser = m_contactService->getClientUser();
            if(NULL == pClientUser.get())
            {
                CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::getMyAccount(), pClientUser == NULL");
                break;
            }
            
            JID = pClientUser->getUri();
        } while (0);
        
        CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::getMyAccount() JID = " << JID);
        return JID;
    }
    
    std::string JabberServiceProvider::getMyDisplayName()
    {
        std::string myDisplayName = "";
        
        do
        {
            if (NULL == m_contactService.get())
            {
                break;
            }

            SMART_PTR_NS::shared_ptr<Contact> pClientUser = m_contactService->getClientUser();
            if(NULL == pClientUser.get())
            {
                CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::getMyDisplayNmae(), pClientUser == NULL");
                break;
            }
            
            myDisplayName = pClientUser->getDisplayName();
        } while (0);
        

        CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::getMyDisplayName(), myDisplayName = " << myDisplayName);
        return myDisplayName;
    }
    
    std::string JabberServiceProvider::getMyEmail()
    {
        std::string email = "";
        
        do
        {
            if (NULL == m_contactService.get())
            {
                break;
            }
            
            SMART_PTR_NS::shared_ptr<Contact> pClientUser = m_contactService->getClientUser();
            if(NULL == pClientUser.get())
            {
                CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::getMyEmail(), pClientUser == NULL");
                break;
            }
            
            email = pClientUser->getEmailAddress();
        } while (0);
        
        
        CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::getMyEmail(), email = " << email);
        return email;
    }
    
    std::string JabberServiceProvider::getDisplayNameByJabberId(std::string JID)
    {
        std::string displayName = "";

        do
        {
            if (NULL == m_contactService.get())
            {
                break;
            }
            
            SMART_PTR_NS::shared_ptr<Contact> contact = m_contactService->FindOrCreateContactByUri(JID);
            if (NULL == contact.get())
            {
                CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::getDisplayNameByJabberId(), contact == NULL");
                break;
            }
            
            displayName = contact->getDisplayName();
        } while (0);
        
        CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::getDisplayNameByJabberId(), displayName = " << displayName << " JID = " << JID);;
        return displayName;
    }
    
    std::string JabberServiceProvider::getEmailByJabberId(std::string JID)
    {
        std::string email = "";
        
        do
        {
            if (NULL == m_contactService.get())
            {
                break;
            }
            
            SMART_PTR_NS::shared_ptr<Contact> contact = m_contactService->FindOrCreateContactByUri(JID);
            if (NULL == contact.get())
            {
                CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::getEmailByJabberId(), contact == NULL");
                break;
            }
            
            email = contact->getEmailAddress();
        } while (0);
        
        CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::getEmailByJabberId(), email = " << email << " JID = " << JID);
        return email;
    }
    
    std::string JabberServiceProvider::getJIDByEmail(std::string email)
    {
        std::string JID = "";
     
        do
        {
            if (NULL == m_contactService.get())
            {
                break;
            }
            
            SMART_PTR_NS::shared_ptr<Contact> contact = m_contactService->FindOrCreateContactByEmailAddress(email);
            if (NULL == contact.get())
            {
                CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::getJIDByEmail(), contact == NULL");
                break;
            }
            
            JID = contact->getUri();
        } while (0);

        CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::getJIDByEmail(), JID = " << JID << " email = " << email);
        return JID;
    }
    
    SMART_PTR_NS::shared_ptr<Contact> JabberServiceProvider::getContact(const std::string& uri)
    {
        SMART_PTR_NS::shared_ptr<Contact> spContact;
        
        if (m_contactService.get() != NULL)
        {
            spContact = m_contactService->FindExistingContactByUri(uri);
            if (spContact == NULL)
            {
                // if we do not already have the contact then this may create one
                // The presence status of the contact may change asyncronously
                // spContact = m_contactProvider->LookupContactByUri(uri);
                spContact = m_contactService->FindOrCreateContactByUri(uri);
            }
        }
        
        return spContact;
    }
    
    IMStateEnum::IMState JabberServiceProvider::getIMState(const std::string& uri)
    {
        SMART_PTR_NS::shared_ptr<Contact> spContact = getContact(uri);
        if (spContact.get() == NULL)
        {
            return IMStateEnum::Unknown;
        }
        
        SMART_PTR_NS::shared_ptr<Presence> spPresence = spContact->getPresence();
        if (spPresence.get() == NULL)
        {
            return IMStateEnum::Unknown;
        }
        
        return spPresence->getInstantMessagingState();
    }
    
    bool JabberServiceProvider::isDSPolicyEnable(std::string uri, int& nPolicyResult)
    {
        CSFUnified::ImPolicyEnum::ImPolicy result = ImPolicyEnum::Denied;
        
        if (NULL != m_presenceService.get())
        {
            result = m_presenceService->ProbePolicy(CSFUnified::ProbePoliciesEnum::DesktopShare, uri);
            nPolicyResult = (int)result;
        }

        return (CSFUnified::ImPolicyEnum::Allowed == result) ? true : false;
    }
    
    void JabberServiceProvider::autoRespPolicy(std::string uri, int& nPolicyResult)
    {
        CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::autoRespPolicy(), no interface for the feature.");
    }
    
    bool JabberServiceProvider::isIMPolicyEnable(const std::string& uri)
    {
        bool isEnable = false;
        if (NULL != m_presenceService.get())
        {
            ImPolicyEnum::ImPolicy policy = m_presenceService->ProbePolicy(ProbePoliciesEnum::InstantMessage, uri);
            if (policy == ImPolicyEnum::Allowed)
            {
                isEnable = true;
            }
        }
        
        return isEnable;
    }
    
    bool JabberServiceProvider::hasDSCapability(std::string uri)
    {
        bool isEnable = false;
        if (NULL != m_presenceService.get())
        {
            isEnable = m_presenceService->ProbeCapabilities(CSFUnified::PresenceCapabilityTypeEnum::XC_DS, uri);
        }
       return isEnable;
    }
    
    bool JabberServiceProvider::hasIMCapability(std::string uri)
    {
        if (NULL == m_presenceService.get())
        {
            return false;
        }
        
        return m_presenceService->ProbeCapabilities(PresenceCapabilityTypeEnum::XC_IM, uri);
    }
    
    bool JabberServiceProvider::isBuddyOnline(std::string uri)
    {
        bool isOnline = false;
        
        do
        {
            if (NULL == m_contactService.get())
            {
                break;
            }
            
            SMART_PTR_NS::shared_ptr<Contact> contact = m_contactService->FindOrCreateContactByUri(uri);
            if (NULL == contact.get())
            {
                CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::isBuddyOnline(), contact == NULL");
                break;
            }
            
            PresenceStateEnum::PresenceState state  = contact->getPresence()->getState();
            if (state != PresenceStateEnum::Unavailable && state != PresenceStateEnum::Unknown)
            {
                isOnline = true;
            }
        } while (0);
        
        return isOnline;
    }
    
    bool JabberServiceProvider::isCupMode()
    {
        bool isCupMode = true;

        do
        {
            if (NULL == m_configService.get())
            {
                break;
            }
            
            SMART_PTR_NS::shared_ptr<ConfigValue> configValue = m_configService->getConfig(L"PresenceServerType");
            if (NULL == configValue.get())
            {
                break;
            }

            std::wstring presenceServerType = configValue->getValue();
            if(!JCFCoreUtils::caseInsensitiveStringEquals(presenceServerType, L"WebEx"))
            {
                break;
            }

            SMART_PTR_NS::shared_ptr<ConfigValue> configValueEx = m_configService->getConfig(L"IsEnforceCupModeForMeetingService");
            if (NULL != configValueEx.get())
            {
                std::wstring temp = configValueEx->getValue();
                if (JCFCoreUtils::caseInsensitiveStringEquals(temp, L"true"))
                {
                    break;
                }
            }
            
            isCupMode = false;
        } while (0);

        CSFLogDebugS(MeetingServiceLogger, "JabberServiceProvider::IsCupMode = " << (isCupMode ? "true" : "false"));
        return isCupMode;
    }
    
    bool JabberServiceProvider::isPhoneMode()
    {
        bool isPhoneMode = false;
        
        do
        {
            if (NULL == m_configService.get())
            {
                break;
            }
            
            SMART_PTR_NS::shared_ptr<ConfigValue> configValue = m_configService->getConfig(L"ProductMode");
            if (NULL == m_configService.get())
            {
                break;
            }
            
            std::wstring productType = configValue->getValue();
            if(!JCFCoreUtils::caseInsensitiveStringEquals(productType, L"Phone_Mode"))
            {
                break;
            }
            
            isPhoneMode = true;
        } while (0);
        
        CSFLogDebugS(MeetingServiceLogger, "JabberServiceProvider::isPhoneMode = " << (isPhoneMode ? "true" : "false"));
        return isPhoneMode;
    }

    bool JabberServiceProvider::isWbxMeetingEnabled()
    {
        bool isEnable = true;
        
        do
        {
            if (NULL == m_configService.get())
            {
                break;
            }

            SMART_PTR_NS::shared_ptr<ConfigValue> configValue = m_configService->getConfig(L"Meetings_Enabled");
            if (NULL == configValue.get())
            {
                break;
            }

            if(JCFCoreUtils::caseInsensitiveStringEquals(configValue->getValue(), L"false"))
            {
				isEnable = false;
                break;
            }
            
        } while (0);
        
        CSFLogDebugS(MeetingServiceLogger, "JabberServiceProvider::IsWbxMeetingEnabled = " << (isEnable ? "true" : "false"));
        return isEnable;
    }
    
    std::string JabberServiceProvider::getDSURL()
    {
        std::string dsURL = "";
        
        do
        {
            if (NULL == m_configService.get())
            {
                break;
            }
            
            SMART_PTR_NS::shared_ptr<ConfigValue> configValue = m_configService->getConfig(L"WebEx_DS_Url");
            if (NULL == configValue.get())
            {
                break;
            }
            
            dsURL = JCFCoreUtils::toString(configValue->getValue());
        } while (0);
        
        return dsURL;
    }

	int JabberServiceProvider::getSignatureMode()
	{
		std::string key = "WBX_MEETING_SDK_SIGNATURE_CHECK_MODE";
		std::string value = getConfigValueByKey(key);

		int ret = 1;

		if (value == "NOT_CHECK")
		{ 
			ret = 0;
		}
		if (value == "CHECK_IMPORT")
		{
			ret = 1;
		}
		else if (value == "CHECK_ALL")
		{
			ret = 2;
		}

		return ret;
	}

    
    std::string JabberServiceProvider::getConfigValueByKey(const std::string key)
    {
        std::string value = "";
        
        do
        {
            if (NULL == m_configService.get())
            {
                break;
            }
            
            SMART_PTR_NS::shared_ptr<ConfigValue> configValue = m_configService->getConfig(JCFCoreUtils::toWideString(key));
            if (NULL == configValue.get())
            {
                break;
            }

            value = JCFCoreUtils::toString(configValue->getValue());
        } while (0);
        
        return value;
    }
    
    void JabberServiceProvider::setConfigValueByKey(const std::string key, const std::string value)
    {
        if (NULL != m_configService.get())
        {
            m_configService->setConfig(JCFCoreUtils::toWideString(key), JCFCoreUtils::toWideString(value));
        }
    }
    
    SMART_PTR_NS::shared_ptr<Credentials> JabberServiceProvider::getMeetingCredential()
    {
        SMART_PTR_NS::shared_ptr<Credentials> cred;
        
        do
        {
            if (NULL == m_unifiedFactory)
            {
                break;
            }
            
            SMART_PTR_NS::shared_ptr<SystemService> systemService = m_unifiedFactory->getService<SystemService>();
            if (NULL == systemService.get())
            {
                break;
            }

            cred = systemService->GetCredentialsForService(MeetingAuthenticatorIdsEnum::MeetingAuthenticator);
        } while (0);
        
        return cred;
    }
    
    void JabberServiceProvider::setMeetingCredential(const std::string userName, const csf::SecureString password)
    {
        csf::ScopedReadRWLock lock(mLock);
        bool isOnDispatcher = m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess();
        
        if(!isOnDispatcher)
        {
            m_unifiedFactory->getServicesDispatcher()->enqueueBlock(boost::bind(&JabberServiceProvider::setMeetingCredential, this, userName, password), "JabberServiceProvider::setMeetingCredential");
            return;
        }

        if (NULL  == m_unifiedFactory)
        {
            return;
        }
        
        SMART_PTR_NS::shared_ptr<SystemService> systemService = m_unifiedFactory->getService<SystemService>();
        if (NULL == systemService.get())
        {
            return;
        }

        SMART_PTR_NS::shared_ptr<Credentials> cred = systemService->GetCredentialsForService(MeetingAuthenticatorIdsEnum::MeetingAuthenticator);
        if (NULL != cred.get() && cred->getIsEditable())
        {
            systemService->SecureUpdateCredentials(userName, password, MeetingAuthenticatorIdsEnum::MeetingAuthenticator, true, false);
        }
    }

	SMART_PTR_NS::shared_ptr<Credentials> JabberServiceProvider::getCustomizedMeetingCredential()
	{
		SMART_PTR_NS::shared_ptr<Credentials> cred;

		do
		{
			if (NULL == m_unifiedFactory)
			{
				break;
			}

			SMART_PTR_NS::shared_ptr<SystemService> systemService = m_unifiedFactory->getService<SystemService>();
			if (NULL == systemService.get())
			{
				break;
			}

			cred = systemService->GetCredentialsForService(MeetingAuthenticatorIdsEnum::CustomizedMeetingAuthenticator);
		} while (0);

		return cred;
	}

	void JabberServiceProvider::setCustomizedMeetingCredential(const std::string userName, const csf::SecureString password)
	{
		csf::ScopedReadRWLock lock(mLock);
		bool isOnDispatcher = m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess();

		if (!isOnDispatcher)
		{
			m_unifiedFactory->getServicesDispatcher()->enqueueBlock(boost::bind(&JabberServiceProvider::setCustomizedMeetingCredential, this, userName, password), "JabberServiceProvider::setCustomizedMeetingCredential");
			return;
		}

		if (NULL == m_unifiedFactory)
		{
			return;
		}

		SMART_PTR_NS::shared_ptr<SystemService> systemService = m_unifiedFactory->getService<SystemService>();
		if (NULL == systemService.get())
		{
			return;
		}

		SMART_PTR_NS::shared_ptr<Credentials> cred = systemService->GetCredentialsForService(MeetingAuthenticatorIdsEnum::CustomizedMeetingAuthenticator);
		if (NULL != cred.get())
		{
			systemService->SecureUpdateCredentials(userName, password, MeetingAuthenticatorIdsEnum::CustomizedMeetingAuthenticator, true, false);
		}
	}

    
    void JabberServiceProvider::addInMeetingStatus()
    {
        CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::addInMeetingStatus");
        
        csf::ScopedReadRWLock lock(m_Lock);
        bool isOnDispatcher = m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess();
        if (isOnDispatcher)
        {
            m_presenceService->UpdateRichPresenceStatus(RichPresenceStatusEnum::InMeeting, true);
        }
        else
        {
            m_unifiedFactory->getServicesDispatcher()->enqueueBlock(boost::bind(&JabberServiceProvider::addInMeetingStatus, this), "MeetingMgrControllerImpl::addInMeetingStatus");
        }
    }
    
    void JabberServiceProvider::removeInMeetingStatus()
    {
        CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::removeInMeetingStatus");
        csf::ScopedReadRWLock lock(m_Lock);
        bool isOnDispatcher = m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess();
        if (isOnDispatcher)
        {
            m_presenceService->UpdateRichPresenceStatus(RichPresenceStatusEnum::InMeeting, false);
        }
        else
        {
            m_unifiedFactory->getServicesDispatcher()->enqueueBlock(boost::bind(&JabberServiceProvider::removeInMeetingStatus, this), "MeetingMgrControllerImpl::removeInMeetingStatus");
        }
    }
    
    void JabberServiceProvider::addInWebExMeetingStatus()
    {
        CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::addInWebExMeetingStatus");
        csf::ScopedReadRWLock lock(m_Lock);
        bool isOnDispatcher = m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess();
        if (isOnDispatcher)
        {
            m_presenceService->UpdateRichPresenceStatus(RichPresenceStatusEnum::InWebexMeeting, true);
        }
        else
        {
            m_unifiedFactory->getServicesDispatcher()->enqueueBlock(boost::bind(&JabberServiceProvider::addInWebExMeetingStatus, this), "MeetingMgrControllerImpl::addInWebExMeetingStatus");
        }
    }
    
    void JabberServiceProvider::removeInWebExMeetingStatus()
    {
        CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::removeInWebExMeetingStatus");
        csf::ScopedReadRWLock lock(m_Lock);
        bool isOnDispatcher = m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess();
        if (isOnDispatcher)
        {
            m_presenceService->UpdateRichPresenceStatus(RichPresenceStatusEnum::InWebexMeeting, false);
        }
        else
        {
            m_unifiedFactory->getServicesDispatcher()->enqueueBlock(boost::bind(&JabberServiceProvider::removeInWebExMeetingStatus, this), "MeetingMgrControllerImpl::removeInWebExMeetingStatus");
        }
    }
    
    void JabberServiceProvider::addSharingInWebExMeetingStatus()
    {
        CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::addSharingInWebExMeetingStatus");
        csf::ScopedReadRWLock lock(m_Lock);
        bool isOnDispatcher = m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess();
        if (isOnDispatcher)
        {
            m_presenceService->UpdateRichPresenceStatus(RichPresenceStatusEnum::InWebexDesktopShare, true);
        }
        else
        {
            m_unifiedFactory->getServicesDispatcher()->enqueueBlock(boost::bind(&JabberServiceProvider::addSharingInWebExMeetingStatus, this), "MeetingMgrControllerImpl::addSharingInWebExMeetingStatus");
        }
    }
    
    void JabberServiceProvider::removeSharingInWebExMeetingStatus()
    {
        CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::removeSharingInWebExMeetingStatus");
        csf::ScopedReadRWLock lock(m_Lock);
        bool isOnDispatcher = m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess();
        if (isOnDispatcher)
        {
            m_presenceService->UpdateRichPresenceStatus(RichPresenceStatusEnum::InWebexDesktopShare, false);
        }
        else
        {
            m_unifiedFactory->getServicesDispatcher()->enqueueBlock(boost::bind(&JabberServiceProvider::removeSharingInWebExMeetingStatus, this), "MeetingMgrControllerImpl::removeSharingInWebExMeetingStatus");
        }
    }
    
    void JabberServiceProvider::addInDSStatus()
    {
        CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::addInDSStatus");
        csf::ScopedReadRWLock lock(m_Lock);
        bool isOnDispatcher = m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess();
        if (isOnDispatcher)
        {
            m_presenceService->UpdateRichPresenceStatus(RichPresenceStatusEnum::InWebexMeeting, true);
        }
        else
        {
            m_unifiedFactory->getServicesDispatcher()->enqueueBlock(boost::bind(&JabberServiceProvider::addInDSStatus, this), "MeetingMgrControllerImpl::addInDSStatus");
        }
    }
    
    void JabberServiceProvider::removeInDSStatus()
    {
        CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::removeInDSStatus");
        csf::ScopedReadRWLock lock(m_Lock);
        bool isOnDispatcher = m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess();
        if (isOnDispatcher)
        {
            m_presenceService->UpdateRichPresenceStatus(RichPresenceStatusEnum::InWebexMeeting, false);
        }
        else
        {
            m_unifiedFactory->getServicesDispatcher()->enqueueBlock(boost::bind(&JabberServiceProvider::removeInDSStatus, this), "MeetingMgrControllerImpl::removeInDSStatus");
        }
    }
    
    void JabberServiceProvider::addSharingInDSStatus()
    {
        CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::addSharingInDSStatus");
        csf::ScopedReadRWLock lock(m_Lock);
        bool isOnDispatcher = m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess();
        if (isOnDispatcher)
        {
            m_presenceService->UpdateRichPresenceStatus(RichPresenceStatusEnum::InWebexDesktopShare, true);
        }
        else
        {
            m_unifiedFactory->getServicesDispatcher()->enqueueBlock(boost::bind(&JabberServiceProvider::addSharingInDSStatus, this), "MeetingMgrControllerImpl::addSharingInDSStatus");
        }
    }
    
    void JabberServiceProvider::removeSharingInDSStatus()
    {
        CSFLogInfoS(MeetingServiceLogger, "JabberServiceProvider::removeSharingInDSStatus");
        csf::ScopedReadRWLock lock(m_Lock);
        bool isOnDispatcher = m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess();
        if (isOnDispatcher)
        {
            m_presenceService->UpdateRichPresenceStatus(RichPresenceStatusEnum::InWebexDesktopShare, false);
        }
        else
        {
            m_unifiedFactory->getServicesDispatcher()->enqueueBlock(boost::bind(&JabberServiceProvider::removeSharingInDSStatus, this), "MeetingMgrControllerImpl::removeSharingInDSStatus");
        }
    }
    
    void JabberServiceProvider::onCapabilityChanged()
    {
        
    }
    
    void JabberServiceProvider::initDefaultService()
    {
        m_contactService = m_unifiedFactory->getService<ContactService>();
        if (NULL == m_contactService.get())
        {
            CSFLogInfoS(MeetingServiceLogger, "MeetingServiceLogger::initDefaultService(), m_contactService == NULL");
        }
        
        m_configService = m_unifiedFactory->getService<ConfigService>();
        if (NULL == m_configService.get())
        {
            CSFLogInfoS(MeetingServiceLogger, "MeetingServiceLogger::initDefaultService(), m_configService == NULL");
        }
        
        m_presenceService = m_unifiedFactory->getService<PresenceService>();
        if (NULL == m_presenceService.get())
        {
            CSFLogInfoS(MeetingServiceLogger, "MeetingServiceLogger::initDefaultService(), m_presenceService == NULL");
        }
    }

//
//	void JabberWerxCommonHelper::InitHelper( SMART_PTR_NS::shared_ptr<UnifiedFactory> UnifiedFactory )
//	{
//		m_UnifiedFactory = UnifiedFactory;
//
//		JabberWerxCommonEvent * pJabberWerxCommonEvent = new JabberWerxCommonEvent();
//		if (pJabberWerxCommonEvent == NULL)
//		{
//			CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::InitHelper.  pJabberWerxCommonEvent == NULL");
//		}
//		m_ptrJabberWerxCommonEvent.reset(pJabberWerxCommonEvent);
//
//		//listen to the AdvancedPresence capability 
//
//		SMART_PTR_NS::shared_ptr<PresenceService> ptrPresenceService = CDependence::getInstance()->GetUnifiedFactory()->getService<PresenceService>();
//		if (NULL != ptrPresenceService)
//		{
//			CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::InitHelper, get PresenceService, to listen to getAdvancedPresenceNotifier");
////			ptrPresenceService->getCapabilities()->getPresenceServiceCapabilitiesNotifiers()->getAdvancedPresenceNotifier()->connect(std::bind(&JabberWerxCommonHelper::OnCapabilityChanged, this));
//		}
//	}
//
//	void JabberWerxCommonHelper::AddMonitor()
//	{
//		CSFLogDebugS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::AddMonitor");
//
//		//CSFSystemMonitor::SystemMonitorPtr pSystemMonitor = CSFSystemMonitor::SystemMonitorFactory::getInstance();
//		//if (pSystemMonitor != NULL)
//		//{
//		//	CSFSystemMonitor::SystemMonitorFactory::getInstance()->addNetworkObserver(m_ptrJabberWerxCommonEvent);
//		//	CSFSystemMonitor::SystemMonitorFactory::getInstance()->addPowerObserver(m_ptrJabberWerxCommonEvent);
//		//}
//		
//		if(PropertyNotifierConnectionPtr l = currentPresenceOptionChanged.lock())
//		{
//			if(l->connected())
//			{
//				l->disconnect();
//			}
//		}
//		if (NULL == m_UnifiedFactory)
//		{
//			CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::AddMonitor.  m_UnifiedFactory == NULL");
//			return;
//		}
//
//		std::string JID = JabberWerxCommonHelper::GetUserName();
//
//		SMART_PTR_NS::shared_ptr<ContactService> ptrCttService = m_UnifiedFactory->getService<ContactService>();
//		if (NULL == ptrCttService)
//		{
//			CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::AddMonitor.  ptrCttService == NULL");
//			return;
//		}
//		SMART_PTR_NS::shared_ptr<CSFUnified::Contact> contact = ptrCttService->FindOrCreateContactByUri(JID);
//
//		if (NULL == contact)
//		{
//			CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::AddMonitor.  ptrCttService == NULL");
//			return;
//		}
//		currentPresenceOptionChanged = contact->getPresence()->getPresenceNotifiers()->getStateNotifier()->connect(std::bind(&JabberWerxCommonEvent::OnPresenceChange, m_ptrJabberWerxCommonEvent.get(), contact));	     
//	}
//
//	void JabberWerxCommonHelper::RemoveMonitor()
//	{
//		//CSFLogDebugS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::RemoveMonitor");
//
//		//CSFSystemMonitor::SystemMonitorPtr pSystemMonitor = CSFSystemMonitor::SystemMonitorFactory::getInstance();
//		//if (pSystemMonitor != NULL)
//		//{
//		//	CSFSystemMonitor::SystemMonitorFactory::getInstance()->removeNetworkObserver(m_ptrJabberWerxCommonEvent);
//		//	CSFSystemMonitor::SystemMonitorFactory::getInstance()->removePowerObserver(m_ptrJabberWerxCommonEvent);
//		//}
//	}
//
//	SMART_PTR_NS::shared_ptr<JabberWerxCommonEvent> JabberWerxCommonHelper::GetJabberCommonEvent()
//	{
//		return m_ptrJabberWerxCommonEvent;
//	}
//
//	bool JabberWerxCommonHelper::IsJIDChanged()
//	{
//		static std::string OldJID = "";
//		std::string NewJID = JabberWerxCommonHelper::GetUserName();
//		if (OldJID != NewJID)
//		{
//			OldJID = NewJID;
//			return true;
//		}
//		else
//		{
//			return false;
//		}
//	}
//
//	void JabberWerxCommonHelper::OnDSTokenValueChanged()
//	{
//		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::OnDSTokenValueChanged.");
//
//		if(PropertyNotifierConnectionPtr l = m_ptrDSTokenChanged.lock())
//		{
//			if(l->connected())
//			{
//				l->disconnect();
//			}
//		}
//		
//		IJabberSDKWrap * pJMSDKWrap = CDependence::getInstance()->GetJabberSDKWrap();
//		if (NULL == pJMSDKWrap)
//		{
//			CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::OnDSTokenValueChanged pJMDependence = NULL");
//			return;
//		}
//
//		IJMCommonEventCallback * pEventCallback = pJMSDKWrap->getCommonEventCallback();
//		if (NULL == pEventCallback)
//		{
//			CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::OnDSTokenValueChanged pEventCallback = NULL");
//			return;
//		}
//
//		std::wstring strToken = L"";
//		std::wstring strUrl = L"";
//
//		SMART_PTR_NS::shared_ptr<PresenceService> ptrDSTokenPS = CDependence::getInstance()->GetUnifiedFactory()->getService<PresenceService>();
//		if (NULL != ptrDSTokenPS)
//		{
//			string strDsToken = ptrDSTokenPS->getDesktopShareToken();
//			if (strDsToken == "")
//			{
//			    CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::OnDSTokenValueChanged Can not get DS_Token.");
//			}
//			else
//			{
//				CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::OnDSTokenValueChanged DS TOKEN = " << strDsToken.c_str());
//				strToken = JCFCoreUtils::toWideString(strDsToken);
//			}
//		}
//
//		strUrl = this->GetDSURL();
//		pEventCallback->OnDSParamReady(strUrl, strToken);
//	}
//
//	void JabberWerxCommonHelper::OnCapabilityChanged()
//	{
////		SMART_PTR_NS::shared_ptr<PresenceService> ptrPS = CDependence::getInstance()->GetUnifiedFactory()->getService<PresenceService>();
//
////		if (NULL != ptrPS && ptrPS->getCapabilities()->getAdvancedPresence()/* && InAMeeting()*/)
////		{
////			//CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::OnCapabilityChanged, getAdvancedPresence:TRUE");
//			//UpdateRichPresenceStatus(InAMeeting);
////		}
//	}
    
    void JabberServiceProvider::SendCmd_CMRMeetingInvitation(const std::string& JID, const std::string& strCmdMsg, const std::string& htmlMsg, const std::string plainMsg)
    {
#ifdef WIN32
        CDependence::getInstance()->GetIMMgr().SendCmd_CMRMeetingInvitation(JID, strCmdMsg, htmlMsg, plainMsg);
#else
        InstantMessageMgr::getInstance()->SendCmd_CMRMeetingInvitation(JID, strCmdMsg, htmlMsg, plainMsg);
#endif
    }
    
    void JabberServiceProvider::startPresenceMonitor()
    {
        bool isOnDispatcher = m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess();
        if(isOnDispatcher)
        {
            _startPresenceMonitor();
        }
        else
        {
            m_unifiedFactory->getServicesDispatcher()->enqueueBlock(boost::bind(&JabberServiceProvider::_startPresenceMonitor, this), "");
        }
    }
    
    void JabberServiceProvider::stopPresenceMonitor()
    {
        bool isOnDispatcher = m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess();
        if (isOnDispatcher)
        {
            _stopPresenceMonitor();
        }
        else
        {
            m_unifiedFactory->getServicesDispatcher()->enqueueBlock(boost::bind(&JabberServiceProvider::_stopPresenceMonitor, this), "");
        }
    }
    
    void JabberServiceProvider::_startPresenceMonitor()
    {
        SMART_PTR_NS::shared_ptr<Contact> spContact = getContact(getMyAccount());
        if (spContact.get() == NULL)
        {
            return;
        }
        
        if (PropertyNotifierConnectionPtr l = m_currentPresenceChange.lock())
        {
            if (l->connected())
            {
                l->disconnect();
            }
        }
        
        m_currentPresenceChange = spContact->getPresence()->getPresenceNotifiers()->getStateNotifier()->connect(boost::bind(&JabberServiceSink::onPresenceChange, &JabberServiceSink::getInstance(), spContact));
    }
    
    void JabberServiceProvider::_stopPresenceMonitor()
    {        
        if (PropertyNotifierConnectionPtr l = m_currentPresenceChange.lock())
        {
            if (l->connected())
            {
                l->disconnect();
            }
        }
    }
}